#  
#  WordCompounder --- Executable file name: GoMusubi
#  Copyright(C) 2021 Kaoru Sagara and Syugo Nakamura 
#  This software is released under any of the GPL (see the file GPL), the LGPL(see the file LGPL), or the BSD License (see the file BSD).

from .base import Compounder
from .consecutive_word_compounder import ConsecutiveWordCompounder


class CommonWordCompounder(Compounder):
  """
  IPAdicとUnidicのそれぞれで結合語(ConsecutiveWordCompounder)を生成し、
  共通する語を最終結合語とみなすルール。

  Args:
      Compounder (Compounder): 結合語生成の基本クラス
  """
  def __init__(self, main_mecab_parser, sub_mecab_parser):
    """
    コンストラクタ。

    Args:
        main_mecab_parser (MecabParser): IPAdicを用いたMeCabパーサ
        sub_mecab_parser (MecabParser): Unidicを用いたMeCabパーサ
    """
    super().__init__()
    self.__main_compounder = ConsecutiveWordCompounder(main_mecab_parser)
    self.__sub_compounder = ConsecutiveWordCompounder(sub_mecab_parser)

  def get_mecab_parser(self):
    """
    メインのMeCabパーサを返す。

    Returns:
        mecab_parser: MeCabパーサ
    """
    return self.__main_compounder.get_mecab_parser()
  
  def process(self):
    """
    合成語生成を実行する。
    """
    super().process()

    # 各辞書を用いて合成語を生成する
    self.main_compounder.process()
    self.sub_compounder.process()
    main_compound_list = self.main_compounder.compound_list
    sub_compound_list = self.sub_compounder.compound_list

    # 共通する合成語を抽出する
    compound_list = list(set(main_compound_list) & set(sub_compound_list))

    # 優先度の高い順にソートする（文字数が多いほど優先度高とする）
    compound_list.sort(key=len, reverse=True)

    self._compound_list = compound_list

  @property
  def main_compounder(self):
    """
    メインのConsecutiveWordCompounderを返す。

    Returns:
        ConsecutiveWordCompounder: IPAdic用ConsecutiveWordCompounder
    """
    return self.__main_compounder

  @property
  def sub_compounder(self):
    """
    サブのConsecutiveWordCompounderを返す。

    Returns:
        ConsecutiveWordCompounder: Unidic用ConsecutiveWordCompounder
    """
    return self.__sub_compounder

  @property
  def sub_mecab_parser(self):
    """
    サブのMecabパーサーを返す。
    """
    return self.__sub_compounder.get_mecab_parser()
